/* High-level minimal WiFi library using esp-idf framework
 * Exposes a few methods, more to write...
 * Assumes simplistic scenario: WiFi STA, WPA2 security via password
 * 
 * TODO:
 * - add in more events to handle, like reconnect, disconnect reason like failed SSID or PASSWORD, etc.
 * - add in wifiscan
 * - add in connecting via BSSID
 * 
 * REV 2025_12_15
 */

#include "6900_wifi.h"
#include "esp_system.h" // various system helper functions, including esp_err_t datatype
#include "esp_wifi.h" // low-level wifi library
#include "esp_log.h" // logging library
#include "esp_event.h" // event loop handler library
#include "nvs_flash.h" //non volatile storage library
// #include "lwip/err.h" // lightweight TCP/IP stack error handling
// #include "lwip/sys.h" // lightweight TCP/IP stack system functions

static const char *TAG = "wifi station"; // tag for logging
static EventGroupHandle_t _s_wifi_event_group;   // declare the event group
static wifi_status_t _wifi_status = WIFI_STA_STOP;

/*
 * Helper function stolen/borrowed from Arduino WiFi library to copy 
 * a const char * into a char *
 */
static size_t _wifi_strncpy(char * dst, const char * src, size_t dst_len){
    if(!dst || !src || !dst_len){
        return 0;
    }
    size_t src_len = strlen(src);
    if(src_len >= dst_len){
        src_len = dst_len;
    } else {
        src_len += 1;
    }
    memcpy(dst, src, src_len);
    return src_len;
}



/**
 * Start Wifi connection with a WPA2 Enterprise AP
 * if passphrase is set the most secure supported mode will be automatically selected
 * @param ssid const char*          Pointer to the SSID string.
 * @param passphrase const char *   Pointer to the password.
 * @return one of the value defined in wl_status_t
 */
wifi_status_t WiFi::begin(const char *ssid, const char *passphrase){

    // Input parameter checks
    if(!ssid || *ssid == 0x00 || strlen(ssid) > 32) {
        ESP_LOGE(TAG, "SSID too long or missing!");
        return WIFI_CONNECT_FAILED;
    }

    if(passphrase && strlen(passphrase) > 64) {
        ESP_LOGE(TAG, "passphrase too long!");
        return WIFI_CONNECT_FAILED;
    }

 
    esp_err_t ret = nvs_flash_init();
    if (ret == ESP_ERR_NVS_NO_FREE_PAGES || ret == ESP_ERR_NVS_NEW_VERSION_FOUND) {
        ESP_ERROR_CHECK(nvs_flash_erase());
        ret = nvs_flash_init();
    }
    ESP_ERROR_CHECK(ret);

    ESP_LOGI(TAG, "ESP_WIFI_MODE_STA");
    
    _wifi_init_sta();   // Initialize STA mode

    ESP_ERROR_CHECK(esp_wifi_set_mode(WIFI_MODE_STA) ); // set up as STA

    // Struct w/ various configuration parameters   
    wifi_config_t wifi_config{};    // initialize and zero out                                                  
    // wifi_config.sta.threshold.authmode = WIFI_AUTH_WPA2_PSK;
    wifi_config.sta.threshold.authmode = WIFI_AUTH_OPEN;    // allow open networks
    wifi_config.sta.pmf_cfg.capable = true;             // default is fine
    wifi_config.sta.pmf_cfg.required = false;           // don't require PMF for OPEN
    if(ssid != NULL && ssid[0] != 0){
        _wifi_strncpy((char*)wifi_config.sta.ssid, ssid, 32);
    	if(passphrase != NULL && passphrase[0] != 0){
    		_wifi_strncpy((char*)wifi_config.sta.password, passphrase, 64);
    	}
    }

    ESP_ERROR_CHECK(esp_wifi_set_config(WIFI_IF_STA, &wifi_config) ); // set config
    ESP_ERROR_CHECK(esp_wifi_start() ); // start wifi

    ESP_LOGI(TAG, "wifi_init_sta finished.");
    if(status() != WIFI_STA_CONNECTED){
    	esp_err_t err = esp_wifi_connect();
    	if(err){
            ESP_LOGE(TAG, "connect failed! 0x%x", err);
            return WIFI_CONNECT_FAILED;
    	}
    }
    return status();
}



/**
 * Start Wifi connection with a WPA2 Enterprise AP
 * if passphrase is set the most secure supported mode will be automatically selected
 * @param ssid const char*          Pointer to the SSID string.
 * @param passphrase const char *   Pointer to the password.
 * @return one of the value defined in wl_status_t
 */
wifi_status_t WiFi::stop() {
    ESP_ERROR_CHECK(esp_wifi_stop());
    return status();
}


/* Event handler for WiFi events
 * This function is called asynchronously when an event is posted to the event 
 * loop. It should be non-blocking and basically only set/clear bits in event group
 * The WIFI_EVENT base has event declarations as shown here:
 * https://docs.espressif.com/projects/esp-idf/en/v5.5.1/esp32c3/api-reference/network/esp_wifi.html#_CPPv412wifi_event_t
 */  
void WiFi::_wifi_event_handler(void* arg, esp_event_base_t event_base,
                                int32_t event_id, void* event_data)
{
    if (event_base == WIFI_EVENT && event_id == WIFI_EVENT_STA_START) {
        xEventGroupSetBits(_s_wifi_event_group, STA_STARTED_BIT);
        _wifi_status = WIFI_STA_DISCONNECTED;
        ESP_LOGI(TAG, "STA START");
    } else if (event_base == WIFI_EVENT && event_id == WIFI_EVENT_STA_STOP) {
        xEventGroupClearBits(_s_wifi_event_group, STA_STARTED_BIT | STA_CONNECTED_BIT | STA_HAS_IP_BIT);
        _wifi_status = WIFI_STA_STOP;
        ESP_LOGI(TAG, "STA STOP");
    } else if (event_base == WIFI_EVENT && event_id == WIFI_EVENT_STA_CONNECTED) {
        xEventGroupSetBits(_s_wifi_event_group, STA_CONNECTED_BIT);
        _wifi_status = WIFI_STA_CONNECTED_NO_IP;
        ESP_LOGI(TAG, "STA CONNECTED");        
    } else if (event_base == WIFI_EVENT && event_id == WIFI_EVENT_STA_DISCONNECTED) {
        xEventGroupClearBits(_s_wifi_event_group, STA_CONNECTED_BIT | STA_HAS_IP_BIT);
        _wifi_status = WIFI_STA_DISCONNECTED;
        wifi_event_sta_disconnected_t* e = (wifi_event_sta_disconnected_t*)(event_data);
        ESP_LOGW(TAG, "STA DISCONNECTED (reason=%d)", e->reason);
        ESP_LOGI(TAG, "retrying connection");         
        esp_wifi_connect(); // auto-retry
    } else if (event_base == IP_EVENT && event_id == IP_EVENT_STA_GOT_IP) {
        ip_event_got_ip_t* event = (ip_event_got_ip_t*) event_data;
        ESP_LOGI(TAG, "got ip:" IPSTR, IP2STR(&event->ip_info.ip));
        xEventGroupSetBits(_s_wifi_event_group, STA_HAS_IP_BIT | STA_CONNECTED_BIT);
        _wifi_status = WIFI_STA_CONNECTED;    
    } else if (event_base == IP_EVENT && event_id == IP_EVENT_STA_LOST_IP) {
         xEventGroupClearBits(_s_wifi_event_group, STA_HAS_IP_BIT);
        _wifi_status = WIFI_STA_CONNECTED_NO_IP;   
    }
}


/* 
 *  Initialize WiFi in STA mode and set up event handlers
 *  @return nothing
*/  
void WiFi::_wifi_init_sta()
{
    if (!_s_wifi_event_group) {
        _s_wifi_event_group = xEventGroupCreate();   // create the event group
    } 

    ESP_ERROR_CHECK(esp_netif_init());  // Initialize network interface
    ESP_ERROR_CHECK(esp_event_loop_create_default()); // Create the default event loop
    esp_netif_create_default_wifi_sta(); // Create netif object w/ default WiFi STA config

    // Initialize WiFi driver w/ default parameters
    wifi_init_config_t cfg = WIFI_INIT_CONFIG_DEFAULT();
    ESP_ERROR_CHECK(esp_wifi_init(&cfg));

    // This next block of code sets up the event handlers
    esp_event_handler_instance_t instance_any_id;       // Create ID to each handler
    esp_event_handler_instance_t instance_got_ip;
    // Register two event handlers to the default loop, one for WiFi, one for IP.
    // WIFI_EVENT and IP_EVENT refer to two different event bases,
    // aka an independent family of events, associated with WIFI or IP.
    // ESP_EVENT_ANY_ID and IP_EVENT... are the event IDs. ANY_ID means that any 
    // WiFi event calls the handler, wheras IP_EVENT_STA_GOT_IP means only that 
    // event calls the handler. Both handlers call the same wifi_event_handler
    ESP_ERROR_CHECK(esp_event_handler_instance_register(WIFI_EVENT,
                                                        ESP_EVENT_ANY_ID,
                                                        &WiFi::_wifi_event_handler,
                                                        NULL,
                                                        &instance_any_id));
    ESP_ERROR_CHECK(esp_event_handler_instance_register(IP_EVENT,
                                                        ESP_EVENT_ANY_ID,
                                                        &WiFi::_wifi_event_handler,
                                                        NULL,
                                                        &instance_got_ip));
}


/**
 * What is the status of the WiFi? 
 * @return returns wl_status_t enum
 */
wifi_status_t WiFi::status(){
    return _wifi_status;
}


/**
 * is STA interface connected?
 * @return true if STA is connected to an AP
 */
bool WiFi::isConnected() 
{
    return (status() == WIFI_STA_CONNECTED);
}


