/* Minimal HTTP Client for ESP-IDF, written in C++
 * For MIT 6.900
 *
 * Joel Voldman
 * Created: Oct 2025
 * Updated: 2026-01-26
 * 
 * Includes a few methods, more to write...
 * Assumes only one instance is ever created, and only one _perform is ever
 * called at once -- not thread-safe!
 * 
 * TODO
 *
 */

#include "6900_http_client.h"
#include "esp_log.h"
#include "esp_crt_bundle.h"
#include <cstring>
#include "esp_tls.h"
#include <sys/param.h>

static const char *TAG = "HTTP_CLIENT";     // TAG for logging
// These variables are defined in the cpp as static so we can use them in the 
// static _http_event_handler function
static int _output_len = 0;           // length of response
static char _content_type[64]{};      // e.g. "application/json; charset=utf-8"
static char _content_encoding[32]{};  // e.g. "gzip"

/**
 * Helper function to understand which HTTP errors warrant a retry
 * @param e esp_err_t error from http request
 * @return True if one of four recoverable errors, False otherwise
 */
static bool _is_recoverable_http_err(esp_err_t e) {
    return (e == ESP_ERR_HTTP_FETCH_HEADER) ||
           (e == ESP_ERR_HTTP_CONNECTION_CLOSED) ||
           (e == ESP_ERR_HTTP_EAGAIN) ||
           (e == ESP_ERR_INVALID_STATE) || 
           (e == ESP_ERR_HTTP_CONNECT);
}


http_client::http_client(const HttpClientConfig& cfg) : _cfg(cfg) {}

/* For destructor, run a cleanup routine */
http_client::~http_client() { 
    _cleanup(); 
}


/**
 * Main public method to send HTTP requests. Handles GET and POST requests
 * Last three parameters are optional, see function prototype
 * @note Not thread-safe!; serialize calls if used from multiple tasks.
 * @note Keeps one persistent handle; switching hosts opens a new connection under the hood.
 * @param url const char* target URL. Must be a null-terminated string valid for the duration of this call.
 * @param method HTTP method of type esp_http_client_method_t, like HTTP_GET_METHOD, HTTP_POST_METHOD
 * @param body Optional payload bytes, will be cast to char* to send
 * @param body_len Optional length of @p body int.
 * @param content_type Optional const char * Content-Type header (e.g., "application/json").
 * @return ESP_OK on success; error code otherwise.
 */
HttpResponse http_client::http_request(const char *url, 
                                    esp_http_client_method_t method, 
                                    const void* body,
                                    int body_len,
                                    const char* content_type) {

    HttpResponse resp; // err defaults to FAIL
    _output_len = 0;        // reset for each new request
    _local_response_buffer[0] = '\0';
    memset(_content_type, 0, sizeof(_content_type));
    memset(_content_encoding, 0, sizeof(_content_encoding));

    // For our first request, we set up _client this way                                    
    if (!_client) {
        _config.url = url;
        _config.method = method;
        _config.event_handler = _http_event_handler;
        _config.user_data = _local_response_buffer;        // Pass address of local buffer to get response
        _config.disable_auto_redirect = true;
        _config.crt_bundle_attach = esp_crt_bundle_attach;
        _config.timeout_ms = _cfg.req_timeout_ms;
        _config.keep_alive_enable = _cfg.keep_alive;
        ESP_LOGI(TAG, "HTTP request with url =>");
        _client = esp_http_client_init(&_config);
        if (!_client) return resp;
    } else {
        // Otherwise we just update relevant parameters so client can try to keep persistent 
        // connection
        esp_http_client_set_user_data(_client, _local_response_buffer); 
        esp_http_client_set_url(_client, url);
        esp_http_client_set_method(_client, method);
    }

    // Now set the connection header, depending on whether we want to keep-alive or not (default)
    if (!_cfg.keep_alive) {
        esp_http_client_set_header(_client, "Connection", "close");
    } else {
        esp_http_client_set_header(_client, "Connection", "keep-alive");
    }


    // Extra work for post method
    if (method == HTTP_METHOD_POST) {
        if (content_type && *content_type) {
            esp_http_client_set_header(_client, "Content-Type", content_type);
        } else {
            ESP_LOGW(TAG, "Trying to POST without a POST Content-Type");
        }
        if (body && body_len > 0) {
            esp_http_client_set_post_field(_client, static_cast<const char*>(body), body_len);
        } else {
            ESP_LOGW(TAG, "Trying to POST without a POST body");            
        }
    } else {
        // ensure no stale body from a previous request
        esp_http_client_set_post_field(_client, nullptr, 0);
    }

    esp_err_t err = esp_http_client_perform(_client);       // does the actual http request

    // If we get an error, and we are trying to keep the connection alive but we got a recoverable error
    // then try again one time
    if (err != ESP_OK && _cfg.keep_alive && _is_recoverable_http_err(err) ) {
        esp_http_client_close(_client);
        // Now go and re-set the various client handle parameters just in case
        // they got screwed up during the _perform() or _close() methods
        // there is no doubt a better way to do this than repeating all the code!
        esp_http_client_set_url(_client, url);        
        esp_http_client_set_method(_client, method);
        if (!_cfg.keep_alive) {
            esp_http_client_set_header(_client, "Connection", "close");
        } else {
            esp_http_client_set_header(_client, "Connection", "keep-alive");
        }        
        if (method == HTTP_METHOD_POST) {
            if (content_type && *content_type) {
                esp_http_client_set_header(_client, "Content-Type", content_type);
            } else {
                ESP_LOGW(TAG, "Trying to POST without a POST Content-Type");
            }
            if (body && body_len > 0) {
                esp_http_client_set_post_field(_client, static_cast<const char*>(body), body_len);
            } else {
                ESP_LOGW(TAG, "Trying to POST without a POST body");            
            }
        } else {
            // ensure no stale body from a previous request
            esp_http_client_set_post_field(_client, nullptr, 0);
        }        

        // reset the response buffer and associated length
        _output_len = 0;
        _local_response_buffer[0] = '\0';        
        esp_http_client_set_user_data(_client, _local_response_buffer); 

        err = esp_http_client_perform(_client);   // try request again
    }

    resp.err = err;        // Now record the error code as it's after all our connect attempts
    resp.status = esp_http_client_get_status_code(_client);

    // Force cleanup on connection errors
    if (resp.err == ESP_ERR_HTTP_CONNECT) {
        ESP_LOGW(TAG, "Connection error detected, forcing cleanup");
        _cleanup();
    }
    
    if (resp.err == ESP_OK) {
        ESP_LOGI(TAG, "HTTP Status = %d, content_length = %" PRId64,
                resp.status,
                esp_http_client_get_content_length(_client));
    } else {
        ESP_LOGW(TAG, "HTTP request failed: err=%s, status=%d", esp_err_to_name(err), resp.status);
        _cleanup();       // If we're getting errors, makes sense to cleanup and start over
    }

    if (!_cfg.keep_alive) _cleanup();     // also cleanup if we are not trying to keep-alive

    return resp; 
}



/**
 * Check whether the returned body is text
 * If content-type is text and the encoding is not compressed/zip, then 
 * we assume body is readable text
 * @return bool true if readable text
 */
bool http_client::body_is_text() {
    if (_content_encoding[0]) {
        if (strcasecmp(_content_encoding, "gzip") == 0 ||
            strcasecmp(_content_encoding, "deflate") == 0 ||
            strcasecmp(_content_encoding, "br") == 0) {
            return false;
        }
    }
    return (_content_type_is_text(_content_type));
}


/**
 * Return the body as text
 * @return const char * body or nullptr if the body is not readable text
 */
const char* http_client::body_cstr() {
    if (body_is_text()) {
        return _local_response_buffer;
    } else {
        return nullptr;
    }
}

/**
 * @return int private variable of the length of the response body
 */
int http_client::body_size()  { return _output_len; }


/**
 * Return the raw response body as an array of bytes. Useful if the body is not text
 * @return const uint8_t* response body
 */
const uint8_t* http_client::body_data() {
    return reinterpret_cast<const uint8_t*>(_local_response_buffer);
}


// *******************************************************
// Private Methods
// *******************************************************


/**
 * Private event handler for HTTP requests
 * Because the underlying function that uses this is ESP-IDF and thus written 
 * in C, this function's signature must be of the form below
 * @param evt pointer to esp_http_client_event_t struct
 * @return esp_err_t type
 */
esp_err_t http_client::_http_event_handler(esp_http_client_event_t *evt) {
    switch (evt->event_id) {
    case HTTP_EVENT_ERROR: {
        ESP_LOGD(TAG, "HTTP_EVENT_ERROR");
        break;
    } case HTTP_EVENT_ON_CONNECTED: {
        ESP_LOGD(TAG, "HTTP_EVENT_ON_CONNECTED");
        break;
    } case HTTP_EVENT_HEADER_SENT: {
        ESP_LOGD(TAG, "HTTP_EVENT_HEADER_SENT");
        break;
    } case HTTP_EVENT_ON_HEADER: {
        if (!evt->header_key || !evt->header_value) break;
        // Parse the header and store in private variables
        if (strcasecmp(evt->header_key, "Content-Type") == 0) {
            strncpy(_content_type, evt->header_value, sizeof(_content_type)-1);
        } else if (strcasecmp(evt->header_key, "Content-Encoding") == 0) {
            strncpy(_content_encoding, evt->header_value, sizeof(_content_encoding)-1);
        }
        ESP_LOGD(TAG, "HTTP_EVENT_ON_HEADER, key=%s, value=%s", evt->header_key,
                evt->header_value);
        break;
    } case HTTP_EVENT_ON_DATA: {
        // This is the main handler. Every time it is called, a chunk of data 
        // will get stored in the buffer user_data, the pointer will get moved 
        // to the end of that buffer to be ready for the next chunk, and 
        // _output_len will keep track of the total size of the buffer. That way
        // if there are multiple NULL terminations or the data is not text, we 
        // can still keep track of the size
        ESP_LOGD(TAG, "HTTP_EVENT_ON_DATA, len=%d", evt->data_len);
        // Clean the buffer in case of a new request
        if (_output_len == 0) {     // this is set to zero in http_request every time there's a new request
            // we are just starting to copy the output data into the use
            memset(evt->user_data, 0, MAX_HTTP_OUTPUT_BUFFER + 1);
        }

        int copy_len = 0;
        // The last byte in evt->user_data is kept for the NULL character in case of out-of-bound access.
        copy_len = MIN(evt->data_len, (MAX_HTTP_OUTPUT_BUFFER - _output_len));
        if (copy_len) {
            memcpy((uint8_t*)evt->user_data + _output_len, evt->data, copy_len);
        }
        _output_len += copy_len;        // increment _output_len
        break;
    } case HTTP_EVENT_ON_FINISH: {
        ESP_LOGI(TAG, "HTTP_EVENT_ON_FINISH");
        break;
    } case HTTP_EVENT_DISCONNECTED: {
            ESP_LOGI(TAG, "HTTP_EVENT_DISCONNECTED");
            int mbedtls_err = 0;
            esp_err_t err = esp_tls_get_and_clear_last_error((esp_tls_error_handle_t)evt->data, &mbedtls_err, NULL);
            if (err != 0) {
                ESP_LOGI(TAG, "Last esp error code: 0x%x", err);
                ESP_LOGI(TAG, "Last mbedtls failure: 0x%x", mbedtls_err);
            }
            break;
    } case HTTP_EVENT_REDIRECT: {
        ESP_LOGD(TAG, "HTTP_EVENT_REDIRECT");
        esp_http_client_set_header(evt->client, "From", "user@example.com");
        esp_http_client_set_header(evt->client, "Accept", "text/html");
        esp_http_client_set_redirection(evt->client);
        break;
    }
    }
    return ESP_OK;
}


/**
 * Check whether the content-type is text
 * Not foolproof, but does major checks
 * Basically, will assume content body is text if the content_type is a 
 * "text-like" aka text, json, xml, javascript, or x-www-form-urlencoded
 * @param ct const char* content-type
 * @return bool true if text
 */
bool http_client::_content_type_is_text(const char* ct) {
    if (!ct || !*ct) return false;  // empty 

    while (*ct == ' ' || *ct == '\t') ++ct;      // strip leading spaces

    // text/*
    if (strncasecmp(ct, "text/", 5) == 0) return true;

    // application/json, application/xml, javascript, form data
    if (strncasecmp(ct, "application/json", 16) == 0) return true;
    if (strncasecmp(ct, "application/xml", 15) == 0) return true;
    if (strncasecmp(ct, "application/javascript", 22) == 0) return true;
    if (strncasecmp(ct, "application/x-www-form-urlencoded", 33) == 0) return true;

    // suffixes +json / +xml
    const char* plus = strrchr(ct, '+');        // find last '+'
    if (plus) {
        if (strcasecmp(plus, "+json") == 0) return true;
        if (strcasecmp(plus, "+xml")  == 0) return true;
    }
    return false;
}

/**
 * Cleanup http client, basically delete _client and set to nullptr
 */
void http_client::_cleanup() {
    if (_client) {
        esp_http_client_cleanup(_client);
        _client = nullptr;
    }
}
